<?php

/**
 * Gallery class.
 *
 */
class LABB_Gallery_Common {

    /**
     * Holds the class object.
     */
    public static $instance;

    /**
     * Primary class constructor.
     *
     */
    public function __construct() {

        add_filter('attachment_fields_to_edit', array($this, 'attachment_field_grid_width'), 10, 2);
        add_filter('attachment_fields_to_save', array($this, 'attachment_field_grid_width_save'), 10, 2);

        // Ajax calls
        add_action('wp_ajax_labb_load_gallery_items', array($this, 'load_gallery_items_callback'));
        add_action('wp_ajax_nopriv_labb_load_gallery_items', array($this, 'load_gallery_items_callback'));

    }

    public function attachment_field_grid_width($form_fields, $post) {
        $form_fields['labb_grid_width'] = array(
            'label' => esc_html__('Masonry Width', 'livemesh-bb-addons'),
            'input' => 'html',
            'html' => '
<select name="attachments[' . $post->ID . '][labb_grid_width]" id="attachments-' . $post->ID . '-labb_grid_width">
  <option ' . selected(get_post_meta($post->ID, 'labb_grid_width', true), "labb-default", false) . ' value="labb-default">' . esc_html__('Default', 'livemesh-bb-addons') . '</option>
  <option ' . selected(get_post_meta($post->ID, 'labb_grid_width', true), "labb-wide", false) . ' value="labb-wide">' . esc_html__('Wide', 'livemesh-bb-addons') . '</option>
</select>',
            'value' => get_post_meta($post->ID, 'labb_grid_width', true),
            'helps' => esc_html__('Width of the image in masonry gallery grid', 'livemesh-bb-addons')
        );

        return $form_fields;
    }

    public function attachment_field_grid_width_save($post, $attachment) {
        if (isset($attachment['labb_grid_width']))
            update_post_meta($post['ID'], 'labb_grid_width', $attachment['labb_grid_width']);

        return $post;
    }

    function load_gallery_items_callback() {
        $items = $this->parse_items($_POST['items']);
        $settings = $this->parse_gallery_settings($_POST['settings']);
        $paged = intval($_POST['paged']);

        echo $this->build_gallery($items, $settings, $paged);

        wp_die();

    }

    function parse_items($items) {

        $parsed_items = array();

        /* Convert to object if array is handed to us - during pagination */
        foreach ($items as $item) {
            // Remove encoded quotes or other characters
            $item['item_name'] = stripslashes($item['item_name']);

            $item['item_description'] = stripslashes($item['item_description']);

            $item['video_link'] = isset($item['video_link']) ? filter_var($item['video_link'], FILTER_DEFAULT) : '';

            $item['mp4_video'] = isset($item['mp4_video']) ? filter_var($item['mp4_video'], FILTER_DEFAULT) : '';

            $item['webm_video'] = isset($item['webm_video']) ? filter_var($item['webm_video'], FILTER_DEFAULT) : '';

            $item['display_video_inline'] = isset($item['display_video_inline']) ? filter_var($item['display_video_inline'], FILTER_VALIDATE_BOOLEAN) : false;

            $parsed_items[] = (object)$item;
        }

        return apply_filters('labb_gallery_parsed_items', $parsed_items, $items);
    }

    function parse_gallery_settings($settings) {

        $s = $settings;

        $s['gallery_class'] = filter_var($s['gallery_class'], FILTER_DEFAULT);

        $s['gallery_id'] = filter_var($s['gallery_id'], FILTER_DEFAULT);

        $s['filterable'] = filter_var($s['filterable'], FILTER_VALIDATE_BOOLEAN);

        $s['per_line'] = filter_var($s['per_line'], FILTER_VALIDATE_INT);

        $s['per_line_tablet'] = filter_var($s['per_line_tablet'], FILTER_VALIDATE_INT);

        $s['per_line_mobile'] = filter_var($s['per_line_mobile'], FILTER_VALIDATE_INT);

        $s['items_per_page'] = filter_var($s['items_per_page'], FILTER_VALIDATE_INT);

        $s['enable_lightbox'] = filter_var($s['enable_lightbox'], FILTER_VALIDATE_BOOLEAN);

        $s['display_item_tags'] = filter_var($s['display_item_tags'], FILTER_VALIDATE_BOOLEAN);

        $s['display_item_title'] = filter_var($s['display_item_title'], FILTER_VALIDATE_BOOLEAN);

        $s = (object)$s; // convert to object since we deal with objects in our gallery methods

        return apply_filters('labb_gallery_parsed_settings', $s);
    }

    function build_gallery($items, $settings, $paged = 1) {

        $output = '';

        $gallery_video = LABB_Gallery_Video::get_instance();

        foreach ($items as $item):

            if (empty($item))
                continue;

            $item_type = $item->item_type;

            // No need to populate anything if no image is provided for the image
            if ($item_type == 'image' && empty($item->item_image))
                continue;

            $style = '';
            if (!empty($item->tags)) {
                $terms = array_map('trim', explode(',', $item->tags));

                foreach ($terms as $term) {
                    // Get rid of spaces before adding the term
                    $style .= ' term-' . preg_replace('/\s+/', '-', $term);
                }
            }
            
            $item_class = 'labb-' . $item_type . '-type';

            $custom_class = get_post_meta($item->item_image, 'labb_grid_width', true);

            if ($custom_class !== '')
                $item_class .= ' ' . $custom_class;

            $entry_output = '<div class="labb-grid-item labb-gallery-item ' . $style . ' ' . $item_class . '">';

            if ($gallery_video->is_inline_video($item, $settings)):

                $entry_output .= $gallery_video->get_inline_video($item, $settings);

            else:

                $entry_image = '<div class="labb-project-image">';

                $size = isset($settings->image_size) ? $settings->image_size : 'large';

                $photo_src = wp_get_attachment_image_src($item->item_image, $size);

                $photo_data = FLBuilderPhoto::get_attachment_data($item->item_image);

                // set params
                $photo_settings = array(
                    'align' => 'center',
                    'link_type' => '',
                    'crop' => $settings->crop,
                    'photo' => $photo_data,
                    'photo_src' => $photo_src[0],
                    'photo_source' => 'library',
                );

                // Display caption of the photo as item name if no name is specified by the user - useful in bulk upload case
                if (empty($item->item_name))
                    $item->item_name = $photo_data->caption;

                // Display description of the photo as item description if no description is specified by the user - useful in bulk upload case
                if (empty($item->item_description))
                    $item->item_description = $photo_data->description;

                if ($gallery_video->is_gallery_video($item, $settings)):

                    $image_html = '';

                    if (isset($photo_data->url)):

                        $image_html = labb_get_image_html($photo_settings);

                    elseif ($item_type == 'youtube' || $item_type == 'vimeo') :

                        $thumbnail_url = $gallery_video->get_video_thumbnail_url($item->video_link, $settings);

                        if (!empty($thumbnail_url)):

                            $image_html = sprintf('<img src="%s" title="%s" alt="%s" class="labb-image"/>', esc_attr($thumbnail_url), esc_html($item->item_name), esc_html($item->item_name));

                        endif;

                    endif;

                    $entry_image .= apply_filters('labb_gallery_video_thumbnail_html', $image_html, $item, $settings);

                else: // item is an image and not a video


                    if (!empty($item->item_link)) {

                        $photo_settings['link_type'] = 'url';

                        $photo_settings['link_url'] = $item->item_link;

                        $photo_settings['link_target'] = $settings->link_target;
                    }

                    $image_html = labb_get_image_html($photo_settings);;

                    $entry_image .= apply_filters('labb_gallery_thumbnail_html', $image_html, $item, $settings);

                endif;

                $image_info = '<div class="labb-image-info">';

                $entry_info = '<div class="labb-entry-info">';

                if ($settings->display_item_title == 'yes'):

                    $entry_title = '<' . $settings->title_tag . ' class="labb-entry-title">';

                    if ($item_type == 'image' && !empty($item->item_link)):

                        $target = $settings->link_target ? 'target="_blank"' : '';

                        $entry_title .= '<a href="' . esc_url($item->item_link)
                            . '" title="' . esc_html($item->item_name)
                            . '"' . $target . '>'
                            . esc_html($item->item_name)
                            . '</a>';

                    else:

                        $entry_title .= esc_html($item->item_name);

                    endif;

                    $entry_title .= '</' . $settings->title_tag . '>';

                    $entry_info .= apply_filters('labb_gallery_entry_info_title', $entry_title, $item, $settings);

                endif;

                if ($gallery_video->is_gallery_video($item, $settings)):

                    $entry_info .= $gallery_video->get_video_lightbox_link($item, $settings);

                endif;

                if ($settings->display_item_tags == 'yes'):

                    $entry_info .= apply_filters('labb_gallery_info_tags', '<span class="labb-terms">' . esc_html($item->tags) . '</span>', $item , $settings);

                endif;

                $entry_info .= '</div><!-- .labb-entry-info -->';

                $image_info .= apply_filters('labb_gallery_entry_info', $entry_info, $item, $settings);

                if ($item_type == 'image' && ($settings->enable_lightbox == 'yes')) :

                    $image_info .= $this->display_image_lightbox_link($item, $settings);

                endif;

                $image_info .= '</div><!-- .labb-image-info -->';

                $entry_image .= apply_filters('labb_gallery_image_info', $image_info, $item, $settings);

                $entry_image .= '</div><!-- .labb-project-image -->';

                $entry_output .= apply_filters('labb_gallery_entry_image', $entry_image, $item, $settings);

            endif;

            /* Allow users to output whatever data they want to after displaying the image - at present
            we don't display anything but things can change */

            $entry_text = '';

            $entry_output .= apply_filters('labb_gallery_entry_text', $entry_text, $item, $settings);

            $entry_output .= '</div>';

            $output .= apply_filters('labb_gallery_item_output', $entry_output, $item, $settings);

        endforeach;

        return $output;

    }

    function display_image_lightbox_link($item, $settings) {

        $photo_data = FLBuilderPhoto::get_attachment_data($item->item_image);

        if ($photo_data) :

            $image_src = $photo_data->url;

            $anchor_type = (empty($item->item_link) ? 'labb-click-anywhere' : 'labb-click-icon');

            $output = '<a class="labb-lightbox-item ' . $anchor_type
                . '" data-fancybox="' . $settings->gallery_class
                . '" href="' . $image_src
                . '" title="' . esc_html($item->item_name)
                . '" data-description="' . wp_kses_post($item->item_description) . '">';

            $output .= '<i class="labb-icon-full-screen"></i>';

            $output .= __('Open larger image for ', 'livemesh-bb-addons') . esc_html($item->item_name);

            $output .= '</a>';

        endif;

        return apply_filters('labb_gallery_image_lightbox_link', $output, $item, $settings);

    }

    function get_gallery_terms($items) {

        $tags = $terms = array();

        foreach ($items as $item) {
            if (!empty($item))
                $tags = array_merge($tags, explode(',', $item->tags));
        }

        // trim whitespaces before applying array_unique
        $tags = array_map('trim', $tags);

        $terms = array_values(array_unique($tags));

        return apply_filters('labb_gallery_terms', $terms);

    }

    function get_items_to_display($items, $paged, $items_per_page) {

        $offset = $items_per_page * ($paged - 1);

        $items = array_slice($items, $offset, $items_per_page);

        return $items;
    }

    function paginate_gallery($items, $settings) {

        $pagination_type = $settings->pagination;

        // no pagination required if option is not chosen by user or if all posts are already displayed
        if ($pagination_type == 'none' || count($items) <= $settings->items_per_page)
            return;

        $max_num_pages = ceil(count($items) / $settings->items_per_page);

        $output = '<div class="labb-pagination">';

        if ($pagination_type == 'load_more') {

            $load_more_output = '<a href="#" class="labb-load-more labb-button">';

            $load_more_output .= esc_html__('Load More', 'livemesh-bb-addons');

            if ($settings->show_remaining == 'yes')
                $load_more_output .= ' - ' . '<span>' . (count($items) - $settings->items_per_page) . '</span>';

            $load_more_output .= '</a>';

            $output .= apply_filters('labb_gallery_load_more_output', $load_more_output, $items, $settings);
        }
        else {
            $page_links = array();

            for ($n = 1; $n <= $max_num_pages; $n++) :
                $page_links[] = '<a class="labb-page-nav labb-numbered' . ($n == 1 ? ' labb-current-page' : '') . '" href="#" data-page="' . $n . '">' . number_format_i18n($n) . '</a>';
            endfor;

            $page_links_output = join("\n", $page_links);

            if (!empty($page_links)) {

                $prev_link = '<a class="labb-page-nav labb-disabled" href="#" data-page="prev"><i class="labb-icon-arrow-left3"></i></a>';

                $next_link = '<a class="labb-page-nav" href="#" data-page="next"><i class="labb-icon-arrow-right3"></i></a>';

                $page_links_output = $prev_link . "\n" . $page_links_output . "\n" . $next_link;

                $output .= apply_filters('labb_gallery_page_links_output', $page_links_output, $items, $settings);
            }
        }

        $output .= '<span class="labb-loading"></span>';

        $output .= '</div>';

        return apply_filters('labb_gallery_pagination', $output, $items, $settings);

    }

    /** Isotope filtering support for Gallery * */

    function get_gallery_terms_filter($terms) {

        $output = '';

        if (!empty($terms)) {

            $output .= '<div class="labb-taxonomy-filter">';

            $output .= '<div class="labb-filter-item segment-0 labb-active"><a data-value="*" href="#">' . esc_html__('All', 'livemesh-bb-addons') . '</a></div>';

            $segment_count = 1;

            foreach ($terms as $term) {

                if (trim($term) !== '') {

                    $output .= '<div class="labb-filter-item segment-' . intval($segment_count) . '"><a href="#" data-value=".term-' . preg_replace('/\s+/', '-', $term) . '" title="' . esc_html__('View all items filed under ', 'livemesh-bb-addons') . esc_attr($term) . '">' . esc_html($term) . '</a></div>';

                    $segment_count++;

                }
            }

            $output .= '</div>';

        }

        return $output;
    }

    /**
     * Returns the singleton instance of the class.
     *
     */
    public static function get_instance() {

        if (!isset(self::$instance) && !(self::$instance instanceof LABB_Gallery_Common)) {
            self::$instance = new LABB_Gallery_Common();
        }

        return self::$instance;

    }

}

// Load the metabox class.
$labb_gallery_common = LABB_Gallery_Common::get_instance();


